/**
* \file: AilAudioSinkImpl.h
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: Android Auto
*
* \author: I. Hayashi / ADITJ/SW / ihayashi@jp.adit-jv.com
*          D. Girnus  / ADITG/ESM / dgirnus@de.adit-jv.com
*
* \copyright (c) 2016-2017 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#ifndef AILAUDIOSINKIMPL_H
#define AILAUDIOSINKIMPL_H

#include <AudioStreaming.h>

#include <aauto/AudioSink.h>
#include <aauto/GalReceiver.h>
#include <aauto/util/scoped_ptr.h>
#include <aauto/util/shared_ptr.h>
#include <aauto/util/WorkQueue.h>
#include <aauto/util/IoBuffer.h>

#include "aauto/AditAudioSink.h"
#include <uspi/ConfigHandle.h>
#include "AilAudioSinkRecord.h"

#include <condition_variable>
#include <mutex>
#include <atomic>
#include <memory>

#define MAX_AUDIO_UNACKED_FRAMES            4096

#define MEDIA_AUDIO_SAMPLING_RATE           48000
#define MEDIA_AUDIO_BITS_PER_SAMPLE         16
#define MEDIA_AUDIO_CHANNELS                2

#define SYSTEM_AUDIO_SAMPLING_RATE          16000
#define SYSTEM_AUDIO_BITS_PER_SAMPLE        16
#define SYSTEM_AUDIO_CHANNELS               1

#define GUIDANCE_AUDIO_SAMPLING_RATE        16000
#define GUIDANCE_AUDIO_BITS_PER_SAMPLE      16
#define GUIDANCE_AUDIO_CHANNELS             1

/* Google Integration guide notes AAC-lc format is:
 * RAW AAC frame without header, decoded buffer size in one frame:
 *      1024 samples for 16kHz,
 *      2048 samples for 48kHz.
 */
#define SAMPLES_IN_ONE_FRAME_FOR_16KHZ      1024    // decoded buffer size in one frame for 16kHz
#define SAMPLES_IN_ONE_FRAME_FOR_48KHZ      2048    // decoded buffer size in one frame for 48kHz


namespace adit { namespace utility { namespace audio {
class Backend;
};};};

namespace adit { namespace aauto
{

/* based on enum AudioStreamType */
static const char *audioSinkName[] =
{
        "AudioSinkUnknown",
        "AudioSinkGuidance",    // AudioStreamType shall be AUDIO_STREAM_GUIDANCE       1
        "AudioSinkSytemAudio",  // AudioStreamType shall be AUDIO_STREAM_SYSTEM_AUDIO   2
        "AudioSinkMedia",       // AudioStreamType shall be AUDIO_STREAM_MEDIA          3
        "AudioSinkTelephony"    // AudioStreamType shall be AUDIO_STREAM_TELEPHONY      4
};

class AilSinkConfig : public uspi::ConfigHandle
{
public:
AilSinkConfig()
{
    mConfSType = "";
    mConfCodec = "";
    mSampleRate = 0;
    mInitToutms = 0;
    mNumBits = 0;
    mNumChannels = 0;
    mUnackedFrames = 0;
    mVerbose = 0;
    mDiscardFrames = 0;
    mDisablePrio = 0;
    mThreadPrio = 0;
    mPeriodms = 0;
    mBufferperiods = 0;
    mSilencems = 0;
    mDevice = "";
    mStreamType = 0;
    mCodec = 0;
    mThresholdMs = 0;
    mAudioRecord = 0;
    mAudioRecordFile = "";
}

bool ResultConfig()
{
    bool isValid = true;
    mConfSType      = getString("audio-sink-stream-type", &isValid, Range(-1, INT_MAX));
    mConfCodec      = getString("audio-sink-stream-codec", &isValid, Range(-1, INT_MAX));
    mSampleRate     = getInt("audio-sink-sampling-rate", &isValid, Range(-1, INT_MAX), MEDIA_AUDIO_SAMPLING_RATE);
    mNumBits        = getInt("audio-sink-bits-per-sample", &isValid, Range(-1, INT_MAX), MEDIA_AUDIO_BITS_PER_SAMPLE);
    mNumChannels    = getInt("audio-sink-channels", &isValid, Range(-1, INT_MAX), MEDIA_AUDIO_CHANNELS);
    mUnackedFrames  = getInt("audio-sink-max-unacked-frames", &isValid, Range(-1, INT_MAX), MAX_AUDIO_UNACKED_FRAMES);
    mDiscardFrames  = getInt("audio-sink-discard-frames", &isValid, Range(0, 1), 0);
    mVerbose        = getInt("enable-verbose-logging", &isValid, Range(0, 1), 0);
    mDisablePrio    = getInt("disable-real-time-priority-audio", &isValid, Range(0, 1), 0);
    mThreadPrio     = getInt("audio-threads-real-time-priority", &isValid, Range(-1, INT_MAX), 61);
    mThresholdMs    = getInt("audio-sink-threshold-ms", &isValid, Range(0, INT_MAX), 0);

    mAudioRecord     = getIntMultiKey("audio-sink-record", "enable", &isValid, Range(-1, 1), 0);
    mAudioRecordFile = getStringMultiKey("audio-sink-record", "filename", &isValid, Range(-1, INT_MAX), "");

    mPeriodms       = getIntMultiKey("audio-playback-device", "periodms", &isValid, Range(-1, INT_MAX));
    mBufferperiods  = getIntMultiKey("audio-playback-device", "bufferperiods", &isValid, Range(-1, INT_MAX), -1);
    mSilencems      = getIntMultiKey("audio-playback-device", "silencems", &isValid, Range(-1, INT_MAX), -1);
    mInitToutms     = getIntMultiKey("audio-playback-device", "inittoutms", &isValid, Range(-1, INT_MAX), 500);
    mDevice         = getStringMultiKey("audio-playback-device", "device", &isValid, Range(-1, INT_MAX));

    return isValid;
}
    string  mConfSType;     // Streaming type
    string  mConfCodec;     // Codec type
    int     mSampleRate;    // Sampling rate
    int     mNumBits;       // Bits per sample
    int     mNumChannels;   // Number of Channels
    int     mUnackedFrames; // unacknowledged Frames

    /* if set true, work queue will be flushed */
    bool    mDiscardFrames;
    bool    mVerbose;       // Verbose log flag
    bool    mDisablePrio;   // Priority Set flag
    int     mThreadPrio;    // Thread Priority
    string  mAilSinkSetting;   // AIL Common settings data

    int     mStreamType;    // Streaming type
    int     mCodec;         // Codec type

    int     mThresholdMs;

    /* Enable record of received audio data */
    int     mAudioRecord;
    string  mAudioRecordFile;

    /* AIL device setting */
    int     mInitToutms;
    int     mPeriodms;
    int     mBufferperiods;
    int     mSilencems;
    string  mDevice;
};

class AilAudioSinkImpl : public IAudioSinkCallbacks, public adit::utility::audio::Streaming
{
public:
    AilAudioSinkImpl(AudioSink* inSink);
    ~AilAudioSinkImpl();

    bool init();
    void shutdown();

    void setConfigItem(string inKey, string inValue);
    void registerCallbacks(IAditAudioSinkCallbacks* inCallbacks);

protected:
    /* IAudioSinkCallbacks */
    void dataAvailableCallback(int32_t sessionId, uint64_t timestamp,
            uint8_t* data, size_t len);
    void dataAvailableCallback(int32_t sessionId, uint64_t timestamp,
            const ::shared_ptr<IoBuffer>& galFrame, uint8_t* data, size_t len);
    int codecConfigCallback(uint8_t* data, size_t len);
    int setupCallback(int mediaCodecType);
    void playbackStartCallback(int32_t sessionId);
    void playbackStopCallback(int32_t sessionId);

    /* AIL streaming callbacks */
    adit::utility::audio::AudioState processing(unsigned char *in, unsigned char **out, uint32_t &frames) final;

    void error(const std::string& data) const final;
    void warning(const std::string& data) const final;
    void info(const std::string& data) const final;
    void debug(const std::string& data) const final;
    adit::utility::eLogLevel checkLogLevel() const final;

    void statistics(const adit::utility::audio::StreamStatistics &status) final;
    void eostreaming(const adit::utility::audio::AudioError error) final;

private:
    class AudioItem
    {
    public:
        AudioItem(AilAudioSinkImpl* sink, int32_t sessionId,
                uint64_t timestamp,const ::shared_ptr<IoBuffer>& data, uint8_t* dataPtr, size_t len)
                : mSink(sink), mSessionId(sessionId), mTimestamp(timestamp),
                  mData(data), mDataPtr(dataPtr), mLen(len), mWasPlayed(false) { }

        AilAudioSinkImpl* mSink;
        int32_t mSessionId;
        uint64_t mTimestamp;
        ::shared_ptr<IoBuffer> mData;
        uint8_t* mDataPtr;
        size_t mLen;
        bool mWasPlayed;
    };

    /**
     * @brief  Helper API to get the AudioStreamType enum to the
     *         audioSinkName string
     */
    const char* sinkName(int AudioStreamType) const { return audioSinkName[AudioStreamType]; };

    /**
     * @brief  Helper API to get the current configuration
     *         and set the internal members (e.g.mSampleRate, etc.)
     */
    bool getCurrConfig(void);

    /**
     * @brief Helper API, called in case of playbackStartCallback from MD.
     *        Prepares AIL device, calculates pre-buffering and creates work queue.
     */
    bool start();
    /**
     * @brief Helper API, called in case of playbackStopCallback from MD or shutdown triggered by Application.
     *        Stop and release AIL device and shuts down the work queue.
     *
     * @param discardRemaining - if true, discards all remaining work items
     */
    void stop(bool discardRemaining);
    /**
     * @brief Helper API, set current AIL processing state depending on
     *        playbackStart-/StopCallback. States are reflected by enum
     *        adit::utility::audio::AudioState
     */
    void setProcessingState(adit::utility::audio::AudioState newState) {
        std::unique_lock<std::mutex> guard(mProcessingStateMutex);
        mProcessingState = newState;
    }
    /**
     * @brief Helper API, return the current AIL processing state.
     */
    adit::utility::audio::AudioState getProcessingState(void) {
        std::unique_lock<std::mutex> guard(mProcessingStateMutex);
        return mProcessingState;
    }


    /* to handle states */
    bool mInitialized;
    bool mStarted;
    bool mShutdown;
    /* to handle strate transisitions */
    std::condition_variable mConditonVar;

    /* to identify session */
    string mSinkName;
    int32_t mSessionId;

    IAditAudioSinkCallbacks* mCallbacks;
    AudioSink* audioSink;

    /* to protect access to deque with AudioItems */
    std::mutex mMutex;
    deque<::shared_ptr<AudioItem> > mAudioItemDeque;

    /* configuration */
    int32_t mThresholdCounter;
    int32_t mPlayItemTimeMs;
    AilSinkConfig  mConfig;

    bool mIsSetThreadParam;
    bool AudioSinkRecordRunning;
    AilAudioSinkRecord mAudioSinkRecord;

    /* AIL */
    std::shared_ptr<adit::utility::audio::Backend> backend;
    adit::utility::audio::AudioState mProcessingState;
    std::mutex mProcessingStateMutex;
};
} } /* namespace adit { namespace aauto */

#endif // AILAUDIOSINKIMPL_H
